<script type="text/javascript">
    (function (factory) {
        if (typeof define === 'function' && define.amd) {
            // AMD. Register as an anonymous module.
            define(['jquery'], factory);
        } else {
            // Browser globals
            factory(jQuery);
        }
    }(function ($) {
        $.timeago = function(timestamp) {
            if (timestamp instanceof Date) {
                return inWords(timestamp);
            } else if (typeof timestamp === "string") {
                return inWords($.timeago.parse(timestamp));
            } else if (typeof timestamp === "number") {
                return inWords(new Date(timestamp));
            } else {
                return inWords($.timeago.datetime(timestamp));
            }
        };
        var $t = $.timeago;

        $.extend($.timeago, {
            settings: {
                refreshMillis: 60000,
                allowPast: true,
                allowFuture: false,
                localeTitle: false,
                cutoff: 0,
                strings: {
                    prefixAgo: null,
                    prefixFromNow: null,
                    suffixAgo: "<?php echo __('ago');?>",
                    suffixFromNow: "<?php echo __('from now');?>",
                    inPast: "<?php echo __('any moment now');?>",
                    seconds: "<?php echo __('Just now');?>",
                    minute: "<?php echo __('about a minute ago');?>",
                    minutes: "<?php echo __('%d minutes ago');?>",
                    hour: "<?php echo __('about an hour ago');?>",
                    hours: "<?php echo __('%d hours ago');?>",
                    day: "<?php echo __('a day ago');?>",
                    days: "<?php echo __('%d days ago');?>",
                    month: "<?php echo __('about a month ago');?>",
                    months: "<?php echo __('%d months ago');?>",
                    year: "<?php echo __('about a year ago');?>",
                    years: "<?php echo __('%d years ago');?>",
                    wordSeparator: " ",
                    numbers: []
                }
            },

            inWords: function(distanceMillis) {
                if(!this.settings.allowPast && ! this.settings.allowFuture) {
                    throw 'timeago allowPast and allowFuture settings can not both be set to false.';
                }

                var $l = this.settings.strings;
                var prefix = $l.prefixAgo;
                var suffix = $l.suffixAgo;
                if (this.settings.allowFuture) {
                    if (distanceMillis < 0) {
                        prefix = $l.prefixFromNow;
                        suffix = $l.suffixFromNow;
                    }
                }

                if(!this.settings.allowPast && distanceMillis >= 0) {
                    return this.settings.strings.inPast;
                }

                var seconds = Math.abs(distanceMillis) / 1000;
                var minutes = seconds / 60;
                var hours = minutes / 60;
                var days = hours / 24;
                var years = days / 365;

                function substitute(stringOrFunction, number) {
                    var string = $.isFunction(stringOrFunction) ? stringOrFunction(number, distanceMillis) : stringOrFunction;
                    var value = ($l.numbers && $l.numbers[number]) || number;
                    return string.replace(/%d/i, value);
                }

                var words = seconds < 45 && substitute($l.seconds, Math.floor(seconds)) ||
                    seconds < 90 && substitute($l.minute, 1) ||
                    minutes < 45 && substitute($l.minutes, Math.floor(minutes)) ||
                    minutes < 90 && substitute($l.hour, 1) ||
                    hours < 24 && substitute($l.hours, Math.floor(hours)) ||
                    hours < 42 && substitute($l.day, 1) ||
                    days < 30 && substitute($l.days, Math.floor(days)) ||
                    days < 45 && substitute($l.month, 1) ||
                    days < 365 && substitute($l.months, Math.floor(days / 30)) ||
                    years < 1.5 && substitute($l.year, 1) ||
                    substitute($l.years, Math.floor(years));

                var separator = $l.wordSeparator || "";
                if ($l.wordSeparator === undefined) { separator = " "; }
                
                 return $.trim([prefix, words].join(separator));

            },

            parse: function(iso8601) {
                var s = $.trim(iso8601);
                s = s.replace(/\.\d+/,""); // remove milliseconds
                s = s.replace(/-/,"/").replace(/-/,"/");
                s = s.replace(/T/," ").replace(/Z/," UTC");
                s = s.replace(/([\+\-]\d\d)\:?(\d\d)/," $1$2"); // -04:00 -> -0400
                s = s.replace(/([\+\-]\d\d)$/," $100"); // +09 -> +0900
                return new Date(s);
            },
            datetime: function(elem) {
                var iso8601 = $t.isTime(elem) ? $(elem).attr("datetime") : $(elem).attr("title");
                return $t.parse(iso8601);
            },
            isTime: function(elem) {
                // jQuery's `is()` doesn't play well with HTML5 in IE
                if( $(elem).hasClass('age') ) {
                    this.settings.strings.years = '%d';
                }else{
                    this.settings.strings.years = '<?php echo __('%d years ago');?>';
                }
                return $(elem).get(0).tagName.toLowerCase() === "time"; // $(elem).is("time");
            }
        });

        // functions that can be called via $(el).timeago('action')
        // init is default when no action is given
        // functions are called with context of a single element
        var functions = {
            init: function(){
                var refresh_el = $.proxy(refresh, this);
                refresh_el();
                var $s = $t.settings;
                if ($s.refreshMillis > 0) {
                    this._timeagoInterval = setInterval(refresh_el, $s.refreshMillis);
                }
            },
            update: function(time){
                var parsedTime = $t.parse(time);
                $(this).data('timeago', { datetime: parsedTime });
                if($t.settings.localeTitle) $(this).attr("title", parsedTime.toLocaleString());
                refresh.apply(this);
            },
            updateFromDOM: function(){
                $(this).data('timeago', { datetime: $t.parse( $t.isTime(this) ? $(this).attr("datetime") : $(this).attr("title") ) });
                refresh.apply(this);
            },
            dispose: function () {
                if (this._timeagoInterval) {
                    window.clearInterval(this._timeagoInterval);
                    this._timeagoInterval = null;
                }
            }
        };

        $.fn.timeago = function(action, options) {
            var fn = action ? functions[action] : functions.init;
            if(!fn){
                throw new Error("Unknown function name '"+ action +"' for timeago");
            }
            // each over objects here and call the requested function
            this.each(function(){
                fn.call(this, options);
            });
            return this;
        };

        function refresh() {
            var data = prepareData(this);
            var $s = $t.settings;

            if (!isNaN(data.datetime)) {
                if ( $s.cutoff == 0 || Math.abs(distance(data.datetime)) < $s.cutoff) {
                    $(this).text(inWords(data.datetime));
                }
            }
            return this;
        }

        function prepareData(element) {
            element = $(element);
            if (!element.data("timeago")) {
                element.data("timeago", { datetime: $t.datetime(element) });
                var text = $.trim(element.text());
                if ($t.settings.localeTitle) {
                    element.attr("title", element.data('timeago').datetime.toLocaleString());
                } else if (text.length > 0 && !($t.isTime(element) && element.attr("title"))) {
                    element.attr("title", text);
                }
            }
            return element.data("timeago");
        }

        function inWords(date) {
            return $t.inWords(distance(date));
        }

        function distance(date) {
            return (new Date().getTime() - date.getTime());
        }

        // fix for IE6 suckage
        document.createElement("abbr");
        document.createElement("time");
    }));


    $(function () {
        setInterval(function () {
            if ( $('.ajax-time').length > 0) {
                $('.ajax-time').timeago().removeClass('.ajax-time');
            }
        }, 600);
    });
</script>